/*
 * jcomapi.c
 *
 * Copyright (C) 1994-1997, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains application interface routines that are used for both
 * compression and decompression.
 */

#define JPEG_INTERNALS
#include "jinclude.h"
#include "jpeglib.h"


GLOBAL(void)
jpeg_CreateStruct (jpeg_struct_ptr cinfo, bool isDecompress)
{
	int i;

  /* Guard against version mismatches between library and caller. */
	cinfo->mem = NULL;		/* so jpeg_destroy knows mem mgr not called */
 

  /* For debugging purposes, we zero the whole master structure.
   * But the application has already set the err pointer, and may have set
   * client_data, so we have to save and restore those fields.
   * Note: if application hasn't set client_data, tools like Purify may
   * complain here.
   */

	struct jpeg_error_mgr * err = cinfo->err;
	void * client_data = cinfo->client_data; /* ignore Purify complaint here */
		MEMZERO(cinfo, SIZEOF(struct jpeg_struct));
	cinfo->err = err;
	cinfo->client_data = client_data;
	/* Initialize a memory manager instance for this object */
	jinit_memory_mgr((j_common_ptr) cinfo);

	/* Zero out pointers to permanent structures. */
	cinfo->progress = NULL;
	cinfo->dest = NULL;
	cinfo->src = NULL;

	cinfo->comp_info = NULL;

	for (i = 0; i < NUM_QUANT_TBLS; i++)
	{
		cinfo->quant_tbl_ptrs[i] = NULL;
	}

	for (i = 0; i < NUM_HUFF_TBLS; i++) 
	{
		cinfo->dc_huff_tbl_ptrs[i] = NULL;
		cinfo->ac_huff_tbl_ptrs[i] = NULL;
	}

	cinfo->script_space = NULL;
	cinfo->marker_list = NULL;

	
	if (isDecompress)
	{
		cinfo->is_decompressor = TRUE;
		jinit_marker_reader((j_decompress_ptr)(cinfo));
		/* And initialize the overall input controller. */
		jinit_input_controller((j_decompress_ptr)(cinfo));

		/* OK, I'm ready */
		cinfo->global_state = DSTATE_START;
	}
	else
	{
		cinfo->is_decompressor = FALSE;
		cinfo->input_gamma = 1.0;	/* in case application forgets */
		 /* OK, I'm ready */
		cinfo->global_state = CSTATE_START;
	}
}

	

/*
 * Abort processing of a JPEG compression or decompression operation,
 * but don't destroy the object itself.
 *
 * For this, we merely clean up all the nonpermanent memory pools.
 * Note that temp files (virtual arrays) are not allowed to belong to
 * the permanent pool, so we will be able to close all temp files here.
 * Closing a data source or destination, if necessary, is the application's
 * responsibility.
 */

GLOBAL(void)
jpeg_abort (j_common_ptr cinfo)
{
  int pool;

  /* Do nothing if called on a not-initialized or destroyed JPEG object. */
  if (cinfo->mem == NULL)
    return;

  /* Releasing pools in reverse order might help avoid fragmentation
   * with some (brain-damaged) malloc libraries.
   */
  for (pool = JPOOL_NUMPOOLS-1; pool > JPOOL_PERMANENT; pool--) {
    (*cinfo->mem->free_pool) (cinfo, pool);
  }

  /* Reset overall state for possible reuse of object */
  if (cinfo->is_decompressor) {
    cinfo->global_state = DSTATE_START;
    /* Try to keep application from accessing now-deleted marker list.
     * A bit kludgy to do it here, but this is the most central place.
     */
    ((j_decompress_ptr) cinfo)->marker_list = NULL;
  } else {
    cinfo->global_state = CSTATE_START;
  }
}


/*
 * Destruction of a JPEG object.
 *
 * Everything gets deallocated except the master jpeg_compress_struct itself
 * and the error manager struct.  Both of these are supplied by the application
 * and must be freed, if necessary, by the application.  (Often they are on
 * the stack and so don't need to be freed anyway.)
 * Closing a data source or destination, if necessary, is the application's
 * responsibility.
 */

GLOBAL(void)
jpeg_destroy (j_common_ptr cinfo)
{
  /* We need only tell the memory manager to release everything. */
  /* NB: mem pointer is NULL if memory mgr failed to initialize. */
  if (cinfo->mem != NULL)
    (*cinfo->mem->self_destruct) (cinfo);
  cinfo->mem = NULL;		/* be safe if jpeg_destroy is called twice */
  cinfo->global_state = 0;	/* mark it destroyed */
}


/*
 * Convenience routines for allocating quantization and Huffman tables.
 * (Would jutils.c be a more reasonable place to put these?)
 */

GLOBAL(JQUANT_TBL *)
jpeg_alloc_quant_table (j_common_ptr cinfo)
{
  JQUANT_TBL *tbl;

  tbl = (JQUANT_TBL *)
    (*cinfo->mem->alloc_small) (cinfo, JPOOL_PERMANENT, SIZEOF(JQUANT_TBL));
  tbl->sent_table = FALSE;	/* make sure this is false in any new table */
  return tbl;
}


GLOBAL(JHUFF_TBL *)
jpeg_alloc_huff_table (j_common_ptr cinfo)
{
  JHUFF_TBL *tbl;

  tbl = (JHUFF_TBL *)
    (*cinfo->mem->alloc_small) (cinfo, JPOOL_PERMANENT, SIZEOF(JHUFF_TBL));
  tbl->sent_table = FALSE;	/* make sure this is false in any new table */
  return tbl;
}
