/*
 * djpeg.c
 *
 * Copyright (C) 1991-1997, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains a command-line user interface for the JPEG decompressor.
 * It should work on any system with Unix- or MS-DOS-style command lines.
 *
 * Two different command line styles are permitted, depending on the
 * compile-time switch TWO_FILE_COMMANDLINE:
 *	djpeg [options]  inputfile outputfile
 *	djpeg [options]  [inputfile]
 * In the second style, output is always to standard output, which you'd
 * normally redirect to a file or pipe to some other program.  Input is
 * either from a named file or from standard input (typically redirected).
 * The second style is convenient on Unix but is unhelpful on systems that
 * don't support pipes.  Also, you MUST use the first style if your system
 * doesn't do binary I/O to stdin/stdout.
 * To simplify script writing, the "-outfile" switch is provided.  The syntax
 *	djpeg [options]  -outfile outputfile  inputfile
 * works regardless of which command line style is used.
 */

#include "cdjpeg.h"		/* Common decls for cjpeg/djpeg applications */
#include "jversion.h"		/* for version message */

#include <ctype.h>		/* to declare isprint() */

#ifdef USE_CCOMMAND		/* command-line reader for Macintosh */
#ifdef __MWERKS__
#include <SIOUX.h>              /* Metrowerks needs this */
#include <console.h>		/* ... and this */
#endif
#ifdef THINK_C
#include <console.h>		/* Think declares it here */
#endif
#endif


/* Create the add-on message string table. */

#define JMESSAGE(code,string)	string ,

static const char * const cdjpeg_message_table[] = {
#include "cderror.h"
  NULL
};


/*
 * This list defines the known output image formats
 * (not all of which need be supported by a given version).
 * You can change the default output format by defining DEFAULT_FMT;
 * indeed, you had better do so if you undefine PPM_SUPPORTED.
 */



#ifndef DEFAULT_FMT		/* so can override from CFLAGS in Makefile */
#define DEFAULT_FMT	FMT_PPM
#endif

static IMAGE_FORMATS requested_fmt;


/*
 * Argument-parsing code.
 * The switch parser is designed to be useful with DOS-style command line
 * syntax, ie, intermixed switches and file names, where only the switches
 * to the left of a given file name affect processing of that file.
 * The main program in this file doesn't actually use this capability...
 */


static const char * progname;	/* program name for error messages */
static char * outfilename;	/* for -outfile switch */


LOCAL(void)
usage (void)
/* complain about bad command line */
{
  fprintf(stderr, "usage: %s [switches] ", progname);
#ifdef TWO_FILE_COMMANDLINE
  fprintf(stderr, "inputfile outputfile\n");
#else
  fprintf(stderr, "[inputfile]\n");
#endif

  fprintf(stderr, "Switches (names may be abbreviated):\n");
  fprintf(stderr, "  -colors N      Reduce image to no more than N colors\n");
  fprintf(stderr, "  -fast          Fast, low-quality processing\n");
  fprintf(stderr, "  -grayscale     Force grayscale output\n");
#ifdef IDCT_SCALING_SUPPORTED
  fprintf(stderr, "  -scale M/N     Scale output image by fraction M/N, eg, 1/8\n");
#endif
#ifdef BMP_SUPPORTED
  fprintf(stderr, "  -bmp           Select BMP output format (Windows style)%s\n",
	  (DEFAULT_FMT == FMT_BMP ? " (default)" : ""));
#endif
#ifdef GIF_SUPPORTED
  fprintf(stderr, "  -gif           Select GIF output format%s\n",
	  (DEFAULT_FMT == FMT_GIF ? " (default)" : ""));
#endif
#ifdef BMP_SUPPORTED
  fprintf(stderr, "  -os2           Select BMP output format (OS/2 style)%s\n",
	  (DEFAULT_FMT == FMT_OS2 ? " (default)" : ""));
#endif
#ifdef PPM_SUPPORTED
  fprintf(stderr, "  -pnm           Select PBMPLUS (PPM/PGM) output format%s\n",
	  (DEFAULT_FMT == FMT_PPM ? " (default)" : ""));
#endif
#ifdef RLE_SUPPORTED
  fprintf(stderr, "  -rle           Select Utah RLE output format%s\n",
	  (DEFAULT_FMT == FMT_RLE ? " (default)" : ""));
#endif
#ifdef TARGA_SUPPORTED
  fprintf(stderr, "  -targa         Select Targa output format%s\n",
	  (DEFAULT_FMT == FMT_TARGA ? " (default)" : ""));
#endif
  fprintf(stderr, "Switches for advanced users:\n");
#ifdef DCT_ISLOW_SUPPORTED
  fprintf(stderr, "  -dct int       Use integer DCT method%s\n",
	  (JDCT_DEFAULT == JDCT_ISLOW ? " (default)" : ""));
#endif
#ifdef DCT_IFAST_SUPPORTED
  fprintf(stderr, "  -dct fast      Use fast integer DCT (less accurate)%s\n",
	  (JDCT_DEFAULT == JDCT_IFAST ? " (default)" : ""));
#endif
#ifdef DCT_FLOAT_SUPPORTED
  fprintf(stderr, "  -dct float     Use floating-point DCT method%s\n",
	  (JDCT_DEFAULT == JDCT_FLOAT ? " (default)" : ""));
#endif
  fprintf(stderr, "  -dither fs     Use F-S dithering (default)\n");
  fprintf(stderr, "  -dither none   Don't use dithering in quantization\n");
  fprintf(stderr, "  -dither ordered  Use ordered dither (medium speed, quality)\n");
#ifdef QUANT_2PASS_SUPPORTED
  fprintf(stderr, "  -map FILE      Map to colors used in named image file\n");
#endif
  fprintf(stderr, "  -nosmooth      Don't use high-quality upsampling\n");
#ifdef QUANT_1PASS_SUPPORTED
  fprintf(stderr, "  -onepass       Use 1-pass quantization (fast, low quality)\n");
#endif
  fprintf(stderr, "  -maxmemory N   Maximum memory to use (in kbytes)\n");
  fprintf(stderr, "  -outfile name  Specify name for output file\n");
  fprintf(stderr, "  -verbose  or  -debug   Emit debug output\n");
  exit(EXIT_FAILURE);
}


LOCAL(int)
parse_switches (j_decompress_ptr cinfo, int argc, char **argv,
		int last_file_arg_seen, boolean for_real)
/* Parse optional switches.
 * Returns argv[] index of first file-name argument (== argc if none).
 * Any file names with indexes <= last_file_arg_seen are ignored;
 * they have presumably been processed in a previous iteration.
 * (Pass 0 for last_file_arg_seen on the first or only iteration.)
 * for_real is FALSE on the first (dummy) pass; we may skip any expensive
 * processing.
 */
{
  int argn;
  char * arg;

  /* Set up default JPEG parameters. */
  requested_fmt = DEFAULT_FMT;	/* set default output file format */
  outfilename = NULL;
  cinfo->err->trace_level = 0;

  /* Scan command line options, adjust parameters */

  for (argn = 1; argn < argc; argn++) {
    arg = argv[argn];
    if (*arg != '-') {
      /* Not a switch, must be a file name argument */
      if (argn <= last_file_arg_seen) {
	outfilename = NULL;	/* -outfile applies to just one input file */
	continue;		/* ignore this name if previously processed */
      }
      break;			/* else done parsing switches */
    }
    arg++;			/* advance past switch marker character */

    if (keymatch(arg, "bmp", 1)) {
      /* BMP output format. */
      requested_fmt = FMT_BMP;

    } else if (keymatch(arg, "colors", 1) || keymatch(arg, "colours", 1) ||
	       keymatch(arg, "quantize", 1) || keymatch(arg, "quantise", 1)) {
      /* Do color quantization. */
      int val;

      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%d", &val) != 1)
	usage();
      cinfo->desired_number_of_colors = val;
      cinfo->quantize_colors = TRUE;

    } else if (keymatch(arg, "dct", 2)) {
      /* Select IDCT algorithm. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (keymatch(argv[argn], "int", 1)) {
	cinfo->dct_method = JDCT_ISLOW;
      } else if (keymatch(argv[argn], "fast", 2)) {
	cinfo->dct_method = JDCT_IFAST;
      } else if (keymatch(argv[argn], "float", 2)) {
	cinfo->dct_method = JDCT_FLOAT;
      } else
	usage();

    } else if (keymatch(arg, "dither", 2)) {
      /* Select dithering algorithm. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (keymatch(argv[argn], "fs", 2)) {
	cinfo->dither_mode = JDITHER_FS;
      } else if (keymatch(argv[argn], "none", 2)) {
	cinfo->dither_mode = JDITHER_NONE;
      } else if (keymatch(argv[argn], "ordered", 2)) {
	cinfo->dither_mode = JDITHER_ORDERED;
      } else
	usage();

    } else if (keymatch(arg, "debug", 1) || keymatch(arg, "verbose", 1)) {
      /* Enable debug printouts. */
      /* On first -d, print version identification */
      static boolean printed_version = FALSE;

      if (! printed_version) {
	fprintf(stderr, "Independent JPEG Group's DJPEG, version %s\n%s\n",
		JVERSION, JCOPYRIGHT);
	printed_version = TRUE;
      }
      cinfo->err->trace_level++;

    } else if (keymatch(arg, "fast", 1)) {
      /* Select recommended processing options for quick-and-dirty output. */
      cinfo->two_pass_quantize = FALSE;
      cinfo->dither_mode = JDITHER_ORDERED;
      if (! cinfo->quantize_colors) /* don't override an earlier -colors */
	cinfo->desired_number_of_colors = 216;
      cinfo->dct_method = JDCT_FASTEST;
      cinfo->do_fancy_upsampling = FALSE;

    } else if (keymatch(arg, "gif", 1)) {
      /* GIF output format. */
      requested_fmt = FMT_GIF;

    } else if (keymatch(arg, "grayscale", 2) || keymatch(arg, "greyscale",2)) {
      /* Force monochrome output. */
      cinfo->out_color_space = JCS_GRAYSCALE;

    } else if (keymatch(arg, "map", 3)) {
      /* Quantize to a color map taken from an input file. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (for_real) {		/* too expensive to do twice! */
#ifdef QUANT_2PASS_SUPPORTED	/* otherwise can't quantize to supplied map */
	FILE * mapfile;

	if ((mapfile = fopen(argv[argn], READ_BINARY)) == NULL) {
	  fprintf(stderr, "%s: can't open %s\n", progname, argv[argn]);
	  exit(EXIT_FAILURE);
	}
	read_color_map(cinfo, mapfile);
	fclose(mapfile);
	cinfo->quantize_colors = TRUE;
#else
	ERREXIT(cinfo, JERR_NOT_COMPILED);
#endif
      }

    } else if (keymatch(arg, "maxmemory", 3)) {
      /* Maximum memory in Kb (or Mb with 'm'). */
      long lval;
      char ch = 'x';

      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%ld%c", &lval, &ch) < 1)
	usage();
      if (ch == 'm' || ch == 'M')
	lval *= 1000L;
      cinfo->mem->max_memory_to_use = lval * 1000L;

    } else if (keymatch(arg, "nosmooth", 3)) {
      /* Suppress fancy upsampling */
      cinfo->do_fancy_upsampling = FALSE;

    } else if (keymatch(arg, "onepass", 3)) {
      /* Use fast one-pass quantization. */
      cinfo->two_pass_quantize = FALSE;

    } else if (keymatch(arg, "os2", 3)) {
      /* BMP output format (OS/2 flavor). */
      requested_fmt = FMT_OS2;

    } else if (keymatch(arg, "outfile", 4)) {
      /* Set output file name. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      outfilename = argv[argn];	/* save it away for later use */

    } else if (keymatch(arg, "pnm", 1) || keymatch(arg, "ppm", 1)) {
      /* PPM/PGM output format. */
      requested_fmt = FMT_PPM;

    } else if (keymatch(arg, "rle", 1)) {
      /* RLE output format. */
      requested_fmt = FMT_RLE;

    } else if (keymatch(arg, "scale", 1)) {
      /* Scale the output image by a fraction M/N. */
      if (++argn >= argc)	/* advance to next argument */
	usage();
      if (sscanf(argv[argn], "%d/%d",
		 &cinfo->scale_num, &cinfo->scale_denom) != 2)
	usage();

    } else if (keymatch(arg, "targa", 1)) {
      /* Targa output format. */
      requested_fmt = FMT_TARGA;

    } else {
      usage();			/* bogus switch */
    }
  }

  return argn;			/* return index of next arg (file name) */
}


/*
 * Marker processor for COM and interesting APPn markers.
 * This replaces the library's built-in processor, which just skips the marker.
 * We want to print out the marker as text, to the extent possible.
 * Note this code relies on a non-suspending data source.
 */

LOCAL(unsigned int)
jpeg_getc (j_decompress_ptr cinfo)
/* Read next byte */
{
  struct jpeg_source_mgr * datasrc = cinfo->src;

  if (datasrc->bytes_in_buffer == 0) {
    if (! (*datasrc->fill_input_buffer) (cinfo))
      ERREXIT(cinfo, JERR_CANT_SUSPEND);
  }
  datasrc->bytes_in_buffer--;
  return GETJOCTET(*datasrc->next_input_byte++);
}


METHODDEF(boolean)
print_text_marker (j_decompress_ptr cinfo)
{
  boolean traceit = (cinfo->err->trace_level >= 1);
  INT32 length;
  unsigned int ch;
  unsigned int lastch = 0;

  length = jpeg_getc(cinfo) << 8;
  length += jpeg_getc(cinfo);
  length -= 2;			/* discount the length word itself */

  if (traceit) {
    if (cinfo->unread_marker == JPEG_COM)
      fprintf(stderr, "Comment, length %ld:\n", (long) length);
    else			/* assume it is an APPn otherwise */
      fprintf(stderr, "APP%d, length %ld:\n",
	      cinfo->unread_marker - JPEG_APP0, (long) length);
  }

  while (--length >= 0) {
    ch = jpeg_getc(cinfo);
    if (traceit) {
      /* Emit the character in a readable form.
       * Nonprintables are converted to \nnn form,
       * while \ is converted to \\.
       * Newlines in CR, CR/LF, or LF form will be printed as one newline.
       */
      if (ch == '\r') {
	fprintf(stderr, "\n");
      } else if (ch == '\n') {
	if (lastch != '\r')
	  fprintf(stderr, "\n");
      } else if (ch == '\\') {
	fprintf(stderr, "\\\\");
      } else if (isprint(ch)) {
	putc(ch, stderr);
      } else {
	fprintf(stderr, "\\%03o", ch);
      }
      lastch = ch;
    }
  }

  if (traceit)
    fprintf(stderr, "\n");

  return TRUE;
}


GLOBAL(int)
deCompressJPG(char* inFileName, char* outFileName, IMAGE_FORMATS formats)
{
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
#ifdef PROGRESS_REPORT
	struct cdjpeg_progress_mgr progress;
#endif
  
	djpeg_dest_ptr dest_mgr = NULL;
	FILE * input_file;
	FILE * output_file;
	JDIMENSION num_scanlines;

  // On Mac, fetch a command line. //



  // Initialize the JPEG decompression object with default error handling. 
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	// Add some application-specific error messages (from cderror.h) 
	jerr.addon_message_table = cdjpeg_message_table;
	jerr.first_addon_message = JMSG_FIRSTADDONCODE;
	jerr.last_addon_message = JMSG_LASTADDONCODE;

  // Insert custom marker processor for COM and APP12.
  // APP12 is used by some digital camera makers for textual info,
  // so we provide the ability to display it as text.
  // If you like, additional APPn marker types can be selected for display,
  // but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   
	jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
	jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  // Now safe to enable signal catcher. 
#ifdef NEED_SIGNAL_CATCHER
	enable_signal_catcher((j_common_ptr) &cinfo);
#endif

  // Scan command line to find file names. 
  // It is convenient to use just one switch-parsing routine, but the switch
  // values read here are ignored; we will rescan the switches after opening
  // the input file.
  // (Exception: tracing level set here controls verbosity for COM markers
  // found during jpeg_read_header...)
   

	///////////////////////////////////////////////////////////
	//just a sec

	if (inFileName==NULL||outFileName==NULL)
	{
		return -1;
	}

	if ((input_file = fopen(inFileName, READ_BINARY)) == NULL) 
	{
		fprintf(stderr, "%s: can't open %s\n", "djpeg", inFileName);
		return -1;
	}

	if ((output_file = fopen(outFileName, WRITE_BINARY)) == NULL) 
	{
		fprintf(stderr, "%s: can't open %s\n", "djpeg", outFileName);
		fclose(input_file);
		return -1;
	}


#ifdef PROGRESS_REPORT
	start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  // Specify data source for decompression 
	jpeg_stdio_src(&cinfo, input_file);

  // Read file header, set default decompression parameters 
	(void) jpeg_read_header(&cinfo, TRUE);

  // Adjust default decompression parameters by re-parsing the options 
  

  // Initialize the output module now to let it override any crucial
   // option settings (for instance, GIF wants to force color quantization).
   
  switch (formats) {
#ifdef BMP_SUPPORTED
  case FMT_BMP:
    dest_mgr = jinit_write_bmp(&cinfo, FALSE);
    break;
  case FMT_OS2:
    dest_mgr = jinit_write_bmp(&cinfo, TRUE);
    break;
#endif
#ifdef GIF_SUPPORTED
  case FMT_GIF:
    dest_mgr = jinit_write_gif(&cinfo);
    break;
#endif
#ifdef PPM_SUPPORTED
  case FMT_PPM:
    dest_mgr = jinit_write_ppm(&cinfo);
    break;
#endif
#ifdef RLE_SUPPORTED
  case FMT_RLE:
    dest_mgr = jinit_write_rle(&cinfo);
    break;
#endif
#ifdef TARGA_SUPPORTED
  case FMT_TARGA:
    dest_mgr = jinit_write_targa(&cinfo);
    break;
#endif
  default:
    ERREXIT(&cinfo, JERR_UNSUPPORTED_FORMAT);
    break;
  }
  dest_mgr->output_file = output_file;

  // Start decompressor 
  (void) jpeg_start_decompress(&cinfo);

  // Write output file header 
  (*dest_mgr->start_output) (&cinfo, dest_mgr);

  // Process data 
  while (cinfo.output_scanline < cinfo.output_height) {
    num_scanlines = jpeg_read_scanlines(&cinfo, dest_mgr->buffer,
					dest_mgr->buffer_height);
    (*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
  }

#ifdef PROGRESS_REPORT
  // Hack: count final pass as done in case finish_output does an extra pass.
  // The library won't have updated completed_passes.
   
	progress.pub.completed_passes = progress.pub.total_passes;
#endif

  // Finish decompression and release memory.
  // I must do it in this order because output module has allocated memory
  // of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   
	(*dest_mgr->finish_output) (&cinfo, dest_mgr);
	(void) jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);

  // Close files, if we opened them 
	if (input_file != stdin)
		fclose(input_file);
	if (output_file != stdout)
		fclose(output_file);

#ifdef PROGRESS_REPORT
	end_progress_monitor((j_common_ptr) &cinfo);
#endif
	// suppress no-return-value warnings 
	return 0;
}


GLOBAL(int)
deCompressJPGf2m(char* inFileName, int maxBufferSize, unsigned char* outputBuffer)
{
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
#ifdef PROGRESS_REPORT
	struct cdjpeg_progress_mgr progress;
#endif
  
	nick_dest_ptr dest_mgr = NULL;
	FILE * input_file;
	JDIMENSION num_scanlines;

  // Initialize the JPEG decompression object with default error handling. 
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	// Add some application-specific error messages (from cderror.h) 
	jerr.addon_message_table = cdjpeg_message_table;
	jerr.first_addon_message = JMSG_FIRSTADDONCODE;
	jerr.last_addon_message = JMSG_LASTADDONCODE;

  // Insert custom marker processor for COM and APP12.
  // APP12 is used by some digital camera makers for textual info,
  // so we provide the ability to display it as text.
  // If you like, additional APPn marker types can be selected for display,
  // but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   
	jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
	jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  // Now safe to enable signal catcher. 
#ifdef NEED_SIGNAL_CATCHER
	enable_signal_catcher((j_common_ptr) &cinfo);
#endif

  // Scan command line to find file names. 
  // It is convenient to use just one switch-parsing routine, but the switch
  // values read here are ignored; we will rescan the switches after opening
  // the input file.
  // (Exception: tracing level set here controls verbosity for COM markers
  // found during jpeg_read_header...)
   

	///////////////////////////////////////////////////////////
	//just a sec

	if (inFileName==NULL)
	{
		return -1;
	}

	if ((input_file = fopen(inFileName, READ_BINARY)) == NULL) 
	{
		fprintf(stderr, "%s: can't open %s\n", "djpeg", inFileName);
		return -1;
	}


#ifdef PROGRESS_REPORT
	start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  // Specify data source for decompression 
	
	jpeg_stdio_src(&cinfo, input_file);
	

  // Read file header, set default decompression parameters 
	(void) jpeg_read_header(&cinfo, TRUE);

	dest_mgr=(nick_dest_ptr)jpeg_memory_dest_decompress(&cinfo, maxBufferSize, outputBuffer);

  // Adjust default decompression parameters by re-parsing the options 
  

  // Initialize the output module now to let it override any crucial
   // option settings (for instance, GIF wants to force color quantization).
   

  //dest_mgr = jinit_write_bmp(&cinfo, FALSE);
 
  //dest_mgr->output_file = NULL;

  // Start decompressor 
  (void) jpeg_start_decompress(&cinfo);

  // Write output file header 

  // Process data 
	int rowWidth, rowCounter=0, offset;
	rowWidth=cinfo.image_width*3;
	

	
	
	while (rowCounter < cinfo.output_height) 
	{
		//offset=(cinfo.image_height-rowCounter-1)*rowWidth;
		offset=rowCounter*rowWidth;
		
		dest_mgr->pub.next_output_byte=outputBuffer+offset;
			
		jpeg_read_scanlines(&cinfo, &(dest_mgr->pub.next_output_byte), 1);
		rowCounter++;
		//memset(outputBuffer+offset-padBytes, 0, padBytes);
	//(*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
	}

#ifdef PROGRESS_REPORT
  // Hack: count final pass as done in case finish_output does an extra pass.
  // The library won't have updated completed_passes.
   
	progress.pub.completed_passes = progress.pub.total_passes;
#endif

  // Finish decompression and release memory.
  // I must do it in this order because output module has allocated memory
  // of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   
	//(*dest_mgr->finish_output) (&cinfo, dest_mgr);
	(void) jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);

  // Close files, if we opened them 
	if (input_file != stdin)
		fclose(input_file);

#ifdef PROGRESS_REPORT
	end_progress_monitor((j_common_ptr) &cinfo);
#endif
	// suppress no-return-value warnings 
	return cinfo.image_height*rowWidth;
}


#if 0
Int JUncompress(PByte DataDst, ULong DataCount, PByte RcvSrc, UInt RcvCount)
{
struct  jpeg_decompress_struct cinfo;
struct  jpeg_error_mgr         jerr;
PBYTE   ImgPtr               = DataDst;
BYTE                           TempLine             [4096];
long                           ReadLen;
    
    ZeroMemory(&cinfo, sizeof(cinfo));
    ZeroMemory(&jerr,  sizeof(jerr));
    // Step 1: allocate and initialize JPEG decompression object 
    // We set up the normal JPEG error routines, then override error_exit. 
    cinfo.err           = jpeg_std_error(&jerr); // Now we can initialize the JPEG compression object. 

    // Now we can initialize the JPEG decompression object.
    jpeg_create_decompress(&cinfo);

    jpeg_memio_src(&cinfo, RcvSrc, RcvCount, &ReadLen);

    jpeg_read_header(&cinfo, TRUE);

    // Step 2: set parameters for decompression
    if(cinfo.jpeg_color_space        != JCS_GRAYSCALE) {
	  cinfo.quantize_colors           = TRUE;
	  cinfo.desired_number_of_colors  = 128;
    }
    // Step 3: Start decompressor 
    jpeg_start_decompress(&cinfo);

    // Step 5: while (scan lines remain to be written) 
    // Here we use the library's state variable cinfo.next_scanline as the
    // loop counter, so that we don't have to keep track ourselves.
    // To keep things simple, we pass one scanline per call; you can pass more if you wish, though.
    JSAMPARRAY        ThisScanline  =  (*cinfo.mem->alloc_sarray)  ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
    DWORD             WidthInByte   = cinfo.output_width * cinfo.output_components;
    
    while(cinfo.output_scanline     < cinfo.output_height) 
    {
       jpeg_read_scanlines(&cinfo,     ThisScanline, 1);
       CopyMemory(ImgPtr, ThisScanline[0],   WidthInByte);
       ImgPtr                      += WidthInByte;
    }

#if 0
    // We may need to do some setup of our own at this point before reading
	// the data.  After jpeg_start_decompress() we have the correct scaled
	// output image dimensions available, as well as the output colormap
	// if we asked for color quantization.
	  
    Create(cinfo.image_width, cinfo.image_height, 8 * cinfo.output_components);

    if(cinfo.jpeg_color_space    == JCS_GRAYSCALE)
	   CreateGrayColourMap(256);
    else
	   SetPalette(cinfo.actual_number_of_colors, cinfo.colormap[0], cinfo.colormap[1], cinfo.colormap[2]);

    // JSAMPLEs per row in output buffer 
    row_stride = cinfo.output_width * cinfo.output_components;

    // Make a one-row-high sample array that will go away when done with image 
    buffer = (*cinfo.mem->alloc_sarray)  ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

    // Step 6: while (scan lines remain to be read) 
    // Here we use the library's state variable cinfo.output_scanline as the
	// loop counter, so that we don't have to keep track ourselves.
	  
    iter.Upset();
    while (cinfo.output_scanline < cinfo.output_height) {
	   (void) jpeg_read_scanlines(&cinfo, buffer, 1);
	   // Assume put_scanline_someplace wants a pointer and sample count. 
	   iter.SetRow(buffer[0], row_stride);
	   iter.PrevRow();
    }
#endif

    // Step 5: Finish decompression
    jpeg_finish_decompress(&cinfo);
    // We can ignore the return value since suspension is not possible with the stdio data source.

    // Step 6: Release JPEG decompression object 
    // This is an important step since it will release a good deal of memory. 
    jpeg_destroy_decompress(&cinfo);

    // At this point you may want to check to see whether any corrupt-data
	// warnings occurred (test whether jerr.pub.num_warnings is nonzero).

    return 1;
}
#endif


GLOBAL(int)
deCompressJPGm2f(int inputBufferSize, unsigned char* inputBuffer, char* outFileName)
{
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
#ifdef PROGRESS_REPORT
	struct cdjpeg_progress_mgr progress;
#endif
  
	djpeg_dest_ptr dest_mgr = NULL;
	FILE * output_file;
	JDIMENSION num_scanlines;

  // Initialize the JPEG decompression object with default error handling. 
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	// Add some application-specific error messages (from cderror.h) 
	jerr.addon_message_table = cdjpeg_message_table;
	jerr.first_addon_message = JMSG_FIRSTADDONCODE;
	jerr.last_addon_message = JMSG_LASTADDONCODE;

  // Insert custom marker processor for COM and APP12.
  // APP12 is used by some digital camera makers for textual info,
  // so we provide the ability to display it as text.
  // If you like, additional APPn marker types can be selected for display,
  // but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   
	jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
	jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  // Now safe to enable signal catcher. 
#ifdef NEED_SIGNAL_CATCHER
	enable_signal_catcher((j_common_ptr) &cinfo);
#endif

  // Scan command line to find file names. 
  // It is convenient to use just one switch-parsing routine, but the switch
  // values read here are ignored; we will rescan the switches after opening
  // the input file.
  // (Exception: tracing level set here controls verbosity for COM markers
  // found during jpeg_read_header...)
   

	///////////////////////////////////////////////////////////
	//just a sec

	if (outFileName==NULL)
	{
		return -1;
	}

	if ((output_file = fopen(outFileName, WRITE_BINARY)) == NULL) 
	{
		fprintf(stderr, "%s: can't open %s\n", "djpeg", outFileName);
		return -1;
	}


#ifdef PROGRESS_REPORT
	start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  // Specify data source for decompression 
	
	//jpeg_stdio_src(&cinfo, input_file);
	

	
	jpeg_memory_src(&cinfo, inputBufferSize, inputBuffer);

  // Read file header, set default decompression parameters 
	(void) jpeg_read_header(&cinfo, TRUE);

  // Adjust default decompression parameters by re-parsing the options 
  

  // Initialize the output module now to let it override any crucial
   // option settings (for instance, GIF wants to force color quantization).
   

  dest_mgr = jinit_write_bmp(&cinfo, FALSE);
  dest_mgr->output_file = output_file;

  // Start decompressor 
  (void) jpeg_start_decompress(&cinfo);

  // Write output file header 
  (*dest_mgr->start_output) (&cinfo, dest_mgr);

  // Process data 
  while (cinfo.output_scanline < cinfo.output_height) {
    num_scanlines = jpeg_read_scanlines(&cinfo, dest_mgr->buffer,
					dest_mgr->buffer_height);
    (*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
  }

#ifdef PROGRESS_REPORT
  // Hack: count final pass as done in case finish_output does an extra pass.
  // The library won't have updated completed_passes.
   
	progress.pub.completed_passes = progress.pub.total_passes;
#endif

  // Finish decompression and release memory.
  // I must do it in this order because output module has allocated memory
  // of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   
	(*dest_mgr->finish_output) (&cinfo, dest_mgr);
	(void) jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);

  // Close files, if we opened them 

	if (output_file != stdout)
		fclose(output_file);

#ifdef PROGRESS_REPORT
	end_progress_monitor((j_common_ptr) &cinfo);
#endif
	// suppress no-return-value warnings 
	return 0;
}



#if 0
//////////////////////////////////////////////////////////////////////////////////////////
Int JUncompress(PByte DataDst, ULong DataCount, PByte RcvSrc, UInt RcvCount)
{
struct  jpeg_decompress_struct cinfo;
struct  jpeg_error_mgr         jerr;
PBYTE   ImgPtr               = DataDst;
BYTE                           TempLine             [4096];
long                           ReadLen;
    
    ZeroMemory(&cinfo, sizeof(cinfo));
    ZeroMemory(&jerr,  sizeof(jerr));
    // Step 1: allocate and initialize JPEG decompression object 
    // We set up the normal JPEG error routines, then override error_exit. 
    cinfo.err           = jpeg_std_error(&jerr); // Now we can initialize the JPEG compression object. 

    // Now we can initialize the JPEG decompression object.
    jpeg_create_decompress(&cinfo);

    jpeg_memio_src(&cinfo, RcvSrc, RcvCount, &ReadLen);

    jpeg_read_header(&cinfo, TRUE);

    // Step 2: set parameters for decompression
    if(cinfo.jpeg_color_space        != JCS_GRAYSCALE) {
	  cinfo.quantize_colors           = TRUE;
	  cinfo.desired_number_of_colors  = 128;
    }
    // Step 3: Start decompressor 
    jpeg_start_decompress(&cinfo);

    // Step 5: while (scan lines remain to be written) 
    // Here we use the library's state variable cinfo.next_scanline as the
    // loop counter, so that we don't have to keep track ourselves.
    // To keep things simple, we pass one scanline per call; you can pass more if you wish, though.
    JSAMPARRAY        ThisScanline  =  (*cinfo.mem->alloc_sarray)  ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
    DWORD             WidthInByte   = cinfo.output_width * cinfo.output_components;
    
    while(cinfo.output_scanline     < cinfo.output_height) 
    {
       jpeg_read_scanlines(&cinfo,     ThisScanline, 1);
       CopyMemory(ImgPtr, ThisScanline[0],   WidthInByte);
       ImgPtr                      += WidthInByte;
    }

#if 0
    // We may need to do some setup of our own at this point before reading
	// the data.  After jpeg_start_decompress() we have the correct scaled
	// output image dimensions available, as well as the output colormap
	// if we asked for color quantization.
	  
    Create(cinfo.image_width, cinfo.image_height, 8 * cinfo.output_components);

    if(cinfo.jpeg_color_space    == JCS_GRAYSCALE)
	   CreateGrayColourMap(256);
    else
	   SetPalette(cinfo.actual_number_of_colors, cinfo.colormap[0], cinfo.colormap[1], cinfo.colormap[2]);

    // JSAMPLEs per row in output buffer 
    row_stride = cinfo.output_width * cinfo.output_components;

    // Make a one-row-high sample array that will go away when done with image 
    buffer = (*cinfo.mem->alloc_sarray)  ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

    // Step 6: while (scan lines remain to be read) 
    // Here we use the library's state variable cinfo.output_scanline as the
	// loop counter, so that we don't have to keep track ourselves.
	  
    iter.Upset();
    while (cinfo.output_scanline < cinfo.output_height) {
	   (void) jpeg_read_scanlines(&cinfo, buffer, 1);
	   // Assume put_scanline_someplace wants a pointer and sample count. 
	   iter.SetRow(buffer[0], row_stride);
	   iter.PrevRow();
    }
#endif

    // Step 5: Finish decompression
    jpeg_finish_decompress(&cinfo);
    // We can ignore the return value since suspension is not possible with the stdio data source.

    // Step 6: Release JPEG decompression object 
    // This is an important step since it will release a good deal of memory. 
    jpeg_destroy_decompress(&cinfo);

    // At this point you may want to check to see whether any corrupt-data
	// warnings occurred (test whether jerr.pub.num_warnings is nonzero).

    return 1;
}
#endif
//////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////

GLOBAL(int)
deCompressJPGm2m(jpeg_descriptor_ptr descriptor)
{
	jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
	nick_dest_ptr dest;
    int padBytes = 0;
	unsigned char*  ptr = NULL;
    djpeg_dest_ptr dest_mgr = NULL;

	/////////////////////////////////////////////
	memset(&cinfo, 0, sizeof(jpeg_decompress_struct));
    memset(&jerr,  0, sizeof(jpeg_error_mgr));
	/////////////////////////////////////////////

	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);
	jpeg_struct_src((j_common_ptr)&cinfo, descriptor->inSize, descriptor->inputBuffer);
	jpeg_struct_dest((j_common_ptr)&cinfo, descriptor->outSize, descriptor->outputBuffer);

	// Add some application-specific error messages (from cderror.h) 
	jerr.addon_message_table = cdjpeg_message_table;
	jerr.first_addon_message = JMSG_FIRSTADDONCODE;
	jerr.last_addon_message = JMSG_LASTADDONCODE;

  // Insert custom marker processor for COM and APP12.
  // APP12 is used by some digital camera makers for textual info,
  // so we provide the ability to display it as text.
  // If you like, additional APPn marker types can be selected for display,
  // but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   
	//jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
	//jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  // Now safe to enable signal catcher. 
#ifdef NEED_SIGNAL_CATCHER
	enable_signal_catcher( &cinfo);
#endif
	jpeg_read_header(&cinfo, TRUE);

/*
	if(cinfo.jpeg_color_space        != JCS_GRAYSCALE) {
	  cinfo.quantize_colors           = TRUE;
	  cinfo.desired_number_of_colors  = 256;
	}
*/	

	dest_mgr = jinit_write_bmp(&cinfo, FALSE);
	jpeg_start_decompress(&cinfo);

	(*dest_mgr->start_output) (&cinfo, dest_mgr);

	// Write output file header 

	// Process data 
	int rowWidth = 0, rowCounter = 0, offset = 0;


	descriptor->imageWidth  = cinfo.image_width;
	descriptor->imageHeight = cinfo.image_height;

	rowWidth = (cinfo.image_width * 3 + 3)&(~3);
	//rowWidth = cinfo.output_width * cinfo.output_components;

	dest=(nick_dest_ptr)(cinfo.dest);

   
	
	if (rowWidth*descriptor->imageHeight <= descriptor->outSize)
	{	
		while (rowCounter < cinfo.output_height) 
		{
			if (descriptor->isUpsidedown)
			{
			   offset = (cinfo.image_height- rowCounter - 1)*rowWidth; 
			}
			else
			{
			   offset = rowCounter*rowWidth;
			}

			if (offset >= descriptor->outSize)
			{
				break;
			}
			memset(descriptor->outputBuffer+offset, 0, rowWidth);

			dest->pub.next_output_byte=descriptor->outputBuffer+offset;
   
			jpeg_read_scanlines(&cinfo, &(dest->pub.next_output_byte), 1);
			ptr = descriptor->outputBuffer+offset;
			if (descriptor->isBGR)
			{
				for (int i=0; i < cinfo.image_width; i++)
				{
					unsigned char temp;
					temp          = ptr[i*3];
					ptr[i*3]      = ptr[i*3+2];					
					ptr[i*3+2] = temp;
				}				
			}
			rowCounter++;
		   //memset(outputBuffer+offset-padBytes, 0, padBytes);
	   //(*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
	   }
   }

#ifdef PROGRESS_REPORT
  // Hack: count final pass as done in case finish_output does an extra pass.
  // The library won't have updated completed_passes.
   
	progress.pub.completed_passes = progress.pub.total_passes;
#endif

  // Finish decompression and release memory.
  // I must do it in this order because output module has allocated memory
  // of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   
	//(*dest_mgr->finish_output) (&cinfo, dest_mgr);
	(void) jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);

 

#ifdef PROGRESS_REPORT
	end_progress_monitor( &cinfo);
#endif
	// suppress no-return-value warnings 
	return rowCounter;
}



#if 0

///////////////////////////////////////////////////////////////////////////////
// JMemSrc.c
// Copyright (C) 1994, Denis Co.

typedef struct {
    struct jpeg_source_mgr pub;	// public fields

    // JOCTET  * buffer;		        // start of buffer
    BYTE    * inmem;		        // source stream
    boolean   start_of_file;	    // have we gotten any data yet?

    long    * RcvLen;   
    long      MaxLen;

} my_source_mgr;
typedef my_source_mgr * my_src_ptr;

//#define INPUT_BUF_SIZE  4096	// choose an efficiently fread'able size
// Initialize source --- called by jpeg_read_header before any data is actually read.
METHODDEF void init_source (j_decompress_ptr cinfo)
{
  my_src_ptr src     = (my_src_ptr) cinfo->src;

  // We reset the empty-input-file flag for each image,
  // but we don't clear the input buffer.
  // This is correct behavior for reading a series of images from one source.
  src->start_of_file = TRUE;
}
// Fill the input buffer --- called whenever buffer is emptied.

// In typical applications, this should read fresh data into the buffer
// (ignoring the current state of next_input_byte & bytes_in_buffer),
// reset the pointer & count to the start of the buffer, and return TRUE
// indicating that the buffer has been reloaded.  It is not necessary to
// fill the buffer entirely, only to obtain at least one more byte.

// There is no such thing as an EOF return.  If the end of the file has been
// reached, the routine has a choice of ERREXIT() or inserting fake data into
// the buffer.  In most cases, generating a warning message and inserting a
// fake EOI marker is the best course of action --- this will allow the
// decompressor to output however much of the image is there.  However,
// the resulting error message is misleading if the real problem is an empty
// input file, so we handle that case specially.

// In applications that need to be able to suspend compression due to input
// not being available yet, a FALSE return indicates that no more data can be
// obtained right now, but more may be forthcoming later.  In this situation,
// the decompressor will return to its caller (with an indication of the
// number of scanlines it has read, if any).  The application should resume
// decompression after it has loaded more data into the input buffer.  Note
// that there are substantial restrictions on the use of suspension --- see
// the documentation.

// When suspending, the decompressor will back up to a convenient restart point
// (typically the start of the current MCU). next_input_byte & bytes_in_buffer
// indicate where the restart point will be if the current call returns FALSE.
// Data beyond this point must be rescanned after resumption, so move it to
// the front of the buffer rather than discarding it.

// DC IN DIRECT MEMORY THIS SHOULD NEVER BE CALLED, BUT IN CASE IT IS...
METHODDEF boolean fill_input_buffer (j_decompress_ptr cinfo)
{
my_src_ptr  src      = (my_src_ptr) cinfo->src;
size_t      nbytes   = 0;

    if(src){
       // nbytes             = JFREAD(src->inmem, src->buffer, INPUT_BUF_SIZE);
       // if(nbytes         <= 0){
       //    if(src->start_of_file){	// Treat empty input file as fatal error
       //       ERREXIT(cinfo, JERR_INPUT_EMPTY);
       //    }
       if(src->pub.bytes_in_buffer != 0){ 
         WARNMS(cinfo,                JWRN_JPEG_EOF);
         // INSERT A FAKE EOI MARKER
         src->inmem[0]              = (JOCTET) 0xFF;
         src->inmem[1]              = (JOCTET) JPEG_EOI;
         nbytes                     = 2;
       } else {
         nbytes                     = src->MaxLen;
       }
       src->pub.next_input_byte = src->inmem;
       src->pub.bytes_in_buffer = nbytes;
       src->start_of_file       = FALSE;

       return TRUE; // DC AND WE ARE ALWAYS DONE...
    }
    return FALSE;
}
// Skip data --- used to skip over a potentially large amount of
// uninteresting data (such as an APPn marker).

// Writers of suspendable-input applications must note that skip_input_data
// is not granted the right to give a suspension return.  If the skip extends
// beyond the data currently in the buffer, the buffer can be marked empty so
// that the next read will cause a fill_input_buffer call that can suspend.
// Arranging for additional bytes to be discarded before reloading the input
// buffer is the application writer's problem.

METHODDEF void skip_input_data (j_decompress_ptr cinfo, long num_bytes)
{
my_src_ptr src = (my_src_ptr) cinfo->src;

  // Just a dumb implementation for now.  Could use fseek() except
  // it doesn't work on pipes.  Not clear that being smart is worth
  // any trouble anyway --- large skips are infrequent.
  if(num_bytes         > 0) {
    while(num_bytes    > (long) src->pub.bytes_in_buffer) {
       num_bytes      -= (long) src->pub.bytes_in_buffer;
       fill_input_buffer(cinfo);
       // note we assume that fill_input_buffer will never return FALSE, so suspension need not be handled.
    }
    src->pub.next_input_byte += (size_t) num_bytes;
    src->pub.bytes_in_buffer -= (size_t) num_bytes;
  }
}
// An additional method that can be provided by data source modules is the
// resync_to_restart method for error recovery in the presence of RST markers.
// For the moment, this source module just uses the default resync method
// provided by the JPEG library.  That method assumes that no backtracking
// is possible.

// Terminate source --- called by jpeg_finish_decompress
// after all data has been read.  Often a no-op.

// NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
// application must deal with any cleanup that should happen even
// for error exit.

METHODDEF void term_source (j_decompress_ptr cinfo)
{
  // no work necessary here
}
// Prepare for input from a stdio stream.
// The caller must have already opened the stream, and is responsible
// for closing it after finishing decompression.

GLOBAL boolean jpeg_memio_src (j_decompress_ptr cinfo, BYTE * inmem,  long MaxLen, long * RcvLen)
{
my_src_ptr src = NULL;

    // The source object and input buffer are made permanent so that a series
    // of JPEG images can be read from the same file by calling jpeg_stdio_src
    // only before the first one.  (If we discarded the buffer at the end of
    // one image, we'd likely lose the start of the next one.)
    // This makes it unsafe to use this manager and a different source
    // manager serially with the same JPEG object.  Caveat programmer.

    if(cinfo->src    == NULL) {	// first time for this JPEG object?
       src            = (my_src_ptr) (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT, SIZEOF(my_source_mgr));
       cinfo->src     = (struct jpeg_source_mgr *)src;
    }
    if((src = (my_src_ptr)cinfo->src) != NULL){
       // src->buffer                     = (JOCTET *) (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT, INPUT_BUF_SIZE * SIZEOF(JOCTET));
       src->pub.init_source            = init_source;
       src->pub.fill_input_buffer      = fill_input_buffer;
       src->pub.skip_input_data        = skip_input_data;
       src->pub.resync_to_restart      = jpeg_resync_to_restart; // use default method
       src->pub.term_source            = term_source;

       src->inmem                      = inmem;
       src->RcvLen                     = RcvLen;
       src->MaxLen                     = MaxLen;

       src->pub.bytes_in_buffer        = 0;      // forces fill_input_buffer on first read
       src->pub.next_input_byte        = 0;      // until buffer loaded
    }
    return ((cinfo->src               != NULL)
    // &&      (src->buffer              != NULL)
    );
}



#endif














/*
 * The main program.
 */

/*
int
main (int argc, char **argv)
{
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
#ifdef PROGRESS_REPORT
  struct cdjpeg_progress_mgr progress;
#endif
  int file_index;
  djpeg_dest_ptr dest_mgr = NULL;
  FILE * input_file;
  FILE * output_file;
  JDIMENSION num_scanlines;

  // On Mac, fetch a command line. //
#ifdef USE_CCOMMAND
  argc = ccommand(&argv);
#endif

  progname = argv[0];
  if (progname == NULL || progname[0] == 0)
    progname = "djpeg";		// in case C library doesn't provide it 

  // Initialize the JPEG decompression object with default error handling. 
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&cinfo);
  // Add some application-specific error messages (from cderror.h) 
  jerr.addon_message_table = cdjpeg_message_table;
  jerr.first_addon_message = JMSG_FIRSTADDONCODE;
  jerr.last_addon_message = JMSG_LASTADDONCODE;

  // Insert custom marker processor for COM and APP12.
  // APP12 is used by some digital camera makers for textual info,
  // so we provide the ability to display it as text.
  // If you like, additional APPn marker types can be selected for display,
  // but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   
  jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
  jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  // Now safe to enable signal catcher. 
#ifdef NEED_SIGNAL_CATCHER
  enable_signal_catcher((j_common_ptr) &cinfo);
#endif

  // Scan command line to find file names. 
  // It is convenient to use just one switch-parsing routine, but the switch
  // values read here are ignored; we will rescan the switches after opening
  // the input file.
  // (Exception: tracing level set here controls verbosity for COM markers
  // found during jpeg_read_header...)
   

  file_index = parse_switches(&cinfo, argc, argv, 0, FALSE);

#ifdef TWO_FILE_COMMANDLINE
  // Must have either -outfile switch or explicit output file name 
  if (outfilename == NULL) {
    if (file_index != argc-2) {
      fprintf(stderr, "%s: must name one input and one output file\n",
	      progname);
      usage();
    }
    outfilename = argv[file_index+1];
  } else {
    if (file_index != argc-1) {
      fprintf(stderr, "%s: must name one input and one output file\n",
	      progname);
      usage();
    }
  }
#else
  // Unix style: expect zero or one file name 
  if (file_index < argc-1) {
    fprintf(stderr, "%s: only one input file\n", progname);
    usage();
  }
#endif // TWO_FILE_COMMANDLINE 

  // Open the input file. 
  if (file_index < argc) {
    if ((input_file = fopen(argv[file_index], READ_BINARY)) == NULL) {
      fprintf(stderr, "%s: can't open %s\n", progname, argv[file_index]);
      exit(EXIT_FAILURE);
    }
  } else {
    // default input file is stdin 
    input_file = read_stdin();
  }

  // Open the output file. 
  if (outfilename != NULL) {
    if ((output_file = fopen(outfilename, WRITE_BINARY)) == NULL) {
      fprintf(stderr, "%s: can't open %s\n", progname, outfilename);
      exit(EXIT_FAILURE);
    }
  } else {
    // default output file is stdout 
    output_file = write_stdout();
  }

#ifdef PROGRESS_REPORT
  start_progress_monitor((j_common_ptr) &cinfo, &progress);
#endif

  // Specify data source for decompression 
  jpeg_stdio_src(&cinfo, input_file);

  // Read file header, set default decompression parameters 
  (void) jpeg_read_header(&cinfo, TRUE);

  // Adjust default decompression parameters by re-parsing the options 
  file_index = parse_switches(&cinfo, argc, argv, 0, TRUE);

  // Initialize the output module now to let it override any crucial
   // option settings (for instance, GIF wants to force color quantization).
   
  switch (requested_fmt) {
#ifdef BMP_SUPPORTED
  case FMT_BMP:
    dest_mgr = jinit_write_bmp(&cinfo, FALSE);
    break;
  case FMT_OS2:
    dest_mgr = jinit_write_bmp(&cinfo, TRUE);
    break;
#endif
#ifdef GIF_SUPPORTED
  case FMT_GIF:
    dest_mgr = jinit_write_gif(&cinfo);
    break;
#endif
#ifdef PPM_SUPPORTED
  case FMT_PPM:
    dest_mgr = jinit_write_ppm(&cinfo);
    break;
#endif
#ifdef RLE_SUPPORTED
  case FMT_RLE:
    dest_mgr = jinit_write_rle(&cinfo);
    break;
#endif
#ifdef TARGA_SUPPORTED
  case FMT_TARGA:
    dest_mgr = jinit_write_targa(&cinfo);
    break;
#endif
  default:
    ERREXIT(&cinfo, JERR_UNSUPPORTED_FORMAT);
    break;
  }
  dest_mgr->output_file = output_file;

  // Start decompressor 
  (void) jpeg_start_decompress(&cinfo);

  // Write output file header 
  (*dest_mgr->start_output) (&cinfo, dest_mgr);

  // Process data 
  while (cinfo.output_scanline < cinfo.output_height) {
    num_scanlines = jpeg_read_scanlines(&cinfo, dest_mgr->buffer,
					dest_mgr->buffer_height);
    (*dest_mgr->put_pixel_rows) (&cinfo, dest_mgr, num_scanlines);
  }

#ifdef PROGRESS_REPORT
  // Hack: count final pass as done in case finish_output does an extra pass.
  // The library won't have updated completed_passes.
   
  progress.pub.completed_passes = progress.pub.total_passes;
#endif

  // Finish decompression and release memory.
  // I must do it in this order because output module has allocated memory
  // of lifespan JPOOL_IMAGE; it needs to finish before releasing memory.
   
  (*dest_mgr->finish_output) (&cinfo, dest_mgr);
  (void) jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);

  // Close files, if we opened them 
  if (input_file != stdin)
    fclose(input_file);
  if (output_file != stdout)
    fclose(output_file);

#ifdef PROGRESS_REPORT
  end_progress_monitor((j_common_ptr) &cinfo);
#endif

  // All done. 
  exit(jerr.num_warnings ? EXIT_WARNING : EXIT_SUCCESS);
  return 0;			// suppress no-return-value warnings 
}
*/
