/*
 * Copyright (C) 2005 Wim Dumon
 *
 * See the LICENSE file for terms of use.
 */

#include <Wt/WText>
#include <Wt/WTable>
#include <Wt/WTableCell>
#include <Wt/WStackedWidget>
#include <Wt/WCssDecorationStyle>
#include <Wt/WApplication>
#include <Wt/WServer>
#include "HangmanGame.h"
#include "LoginWidget.h"
#include "HangmanWidget.h"
#include "HighScoresWidget.h"


HangmanGame::GameChoice HangmanGame::path2Choice(const std::string& path)
{
    if (path.compare("/login") == 0)
    {
        return LOGIN;
    }
    else
    {
        if (path.compare("/play") == 0)
        {
            return PLAY;
        }
        else
        {
            if (path.compare("/score")==0)
            {
                return SCORE;
            }
            else
            {
               return UNKNOWN;
            }
        }
    }
}

std::string HangmanGame::choice2Path(HangmanGame::GameChoice choice)
{
    switch (choice)
    {
    case PLAY:
        return "/play";
    case SCORE:
        return "/score";
    case LOGIN:
    default:
        return "/login";
    }
}

HangmanGame::HangmanGame(WContainerWidget *parent, WApplication* app):
        WTable(parent), App(app)
{
    User = L"guest";
    resize(WLength(100, WLength::Percentage), WLength::Auto);

    Title = new WText(L"A Witty game: Hangman", elementAt(0,0));
    Title->decorationStyle().font().setSize(WFont::XXLarge);
    // Center the title horizontally.
    elementAt(0, 0)->setContentAlignment(AlignTop | AlignCenter);


    MsgText = new WText(L" Gaming Grounds ", elementAt(1, 0));
    elementAt(1, 0)->setContentAlignment(AlignTop | AlignCenter);

    // Element (1,1) holds a stack of widgets with the main content.
    // This is where we switch between Login, Game, and Highscores widgets.
    MainStack = new WStackedWidget(elementAt(2, 0));
    MainStack->setPadding(20);
    Login = new LoginWidget(this);
    MainStack->addWidget(Login);
    Scores = new HighScoresWidget();
    MainStack->addWidget(Scores);
    Game = new HangmanWidget();
    MainStack->addWidget(Game);

    LoginButton = new WPushButton(L" login ", elementAt(3, 0));
    PlayButton =  new WPushButton(L" play ", elementAt(3, 0));
    ScoreButton = new WPushButton(L" score ", elementAt(3, 0));
    LoginButton->clicked().connect(this, &HangmanGame::login);
    PlayButton->clicked().connect(this, &HangmanGame::play);
    ScoreButton->clicked().connect(this, &HangmanGame::score);
    // Center the buttons horizontally.
    elementAt(3, 0)->setContentAlignment(AlignTop | AlignCenter);

    myPath = App->internalPath();

    Path = new WText("Internal Path:"+ myPath, elementAt(4, 0));
    elementAt(4, 0)->setContentAlignment(AlignTop | AlignCenter);
    App->internalPathChanged().connect(this, &HangmanGame::updateInternalPath);
    bLogin = false;
    //App->setInternalPath("/login", false);
    login();
}

void HangmanGame::updateInternalPathByChoice(GameChoice choice)
{
    myPath = choice2Path(choice);
    App->setInternalPath(myPath, false);
    Path->setText("Internal Path:" + myPath);

    switch (choice)
    {
    case SCORE:
        doScore();
        break;
    case LOGIN:
        doLogin();
        break;
    case PLAY:
        doPlay();
        break;
    case UNKNOWN:
        //should not reach here!!!
        throw WServer::Exception("UNKNOWN choice");
        break;
    }
}

bool HangmanGame::stateMachine(const std::string& path, GameChoice& result)
{
    GameChoice current = path2Choice(myPath);
    GameChoice target = path2Choice(path);

    if (bLogin)
    {
        if (target == UNKNOWN)
        {
            result = LOGIN;
        }
        else
        {
            result = target;
        }
    }
    else
    {
       result = LOGIN;
    }
    return current != target && target == result;
}

void HangmanGame::updateInternalPath()
{
    std::string path = App->internalPath();
    GameChoice choice;
    if (stateMachine(path, choice))
    {
        updateInternalPathByChoice(choice);
    }
}

void HangmanGame::doLogin()
{
    MainStack->setCurrentWidget(Login);
    MsgText->setText("login please...");
}

void HangmanGame::doPlay()
{
    MsgText->setText("enjoy the game...");
    MainStack->setCurrentWidget(Game);
    Game->setupGame(User, Dict);
}

void HangmanGame::login()
{
    App->setInternalPath("/login", true);
}

void HangmanGame::play()
{
    App->setInternalPath("/play", true);
}

void HangmanGame::score()
{
    App->setInternalPath("/score", true);
}

void HangmanGame::doScore()
{
    MsgText->setText("check your rank...");
    MainStack->setCurrentWidget(Scores);
    Scores->update(User);
}


